package com.yuda.paperstudentconfig.utils;

import com.yuda.paperstudentconfig.anotation.ExcelColumn;
import org.apache.poi.ss.usermodel.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ExcelImportUtil {

    public static <T> List<T> importExcel(MultipartFile file, Class<T> clazz) throws IOException {
        List<T> resultList = new ArrayList<>();
        
        // 获取Workbook
        Workbook workbook = WorkbookFactory.create(file.getInputStream());
        Sheet sheet = workbook.getSheetAt(0);
        
        // 获取表头映射
        Row headerRow = sheet.getRow(0);
        Map<String, Integer> headerMap = new HashMap<>();
        for (Cell cell : headerRow) {
            headerMap.put(cell.getStringCellValue(), cell.getColumnIndex());
        }
        
        // 遍历数据行
        for (int i = 1; i <= sheet.getLastRowNum(); i++) {
            Row row = sheet.getRow(i);
            T instance = createInstanceFromRow(row, headerMap, clazz);
            if (instance != null) {
                resultList.add(instance);
            }
        }
        
        workbook.close();
        return resultList;
    }

    private static <T> T createInstanceFromRow(Row row, Map<String, Integer> headerMap, Class<T> clazz) {
        try {
            T instance = clazz.getDeclaredConstructor().newInstance();
            
            for (Field field : clazz.getDeclaredFields()) {
                if (field.isAnnotationPresent(ExcelColumn.class)) {
                    ExcelColumn annotation = field.getAnnotation(ExcelColumn.class);
                    String columnName = annotation.value();
                    
                    if (headerMap.containsKey(columnName)) {
                        int columnIndex = headerMap.get(columnName);
                        Cell cell = row.getCell(columnIndex);
                        field.setAccessible(true);
                        field.set(instance, getCellValue(cell, field.getType()));
                    }
                }
            }
            return instance;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    private static Object getCellValue(Cell cell, Class<?> fieldType) {
        if (cell == null) {
            return null;
        }
        switch (cell.getCellType()) {
            case STRING:
                return fieldType == String.class ? cell.getStringCellValue() : null;
            case NUMERIC:
                if (fieldType == Integer.class || fieldType == int.class) {
                    return (int) cell.getNumericCellValue();
                } else if (fieldType == Double.class || fieldType == double.class) {
                    return cell.getNumericCellValue();
                } else if (fieldType == Long.class || fieldType == long.class) {
                    return (long) cell.getNumericCellValue();
                } else if (fieldType == String.class) {
                    // 将数字类型转换为字符串
                    return String.valueOf((long) cell.getNumericCellValue());
                }
                return null;
            case BOOLEAN:
                return fieldType == Boolean.class || fieldType == boolean.class ? cell.getBooleanCellValue() : null;
            default:
                return null;
        }
    }
}